/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This is the main module
 */

#define _GNU_SOURCE

#include "config.h"

#include <stdio.h>
#include <termio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include "updatepass.h"
#include "db_mysql.h"

int
substitute_fld (variable_t *vars, cfg_field_t *fld)
{
  char *str;
  while (fld)
  {
    if (fld->rvalue)
    {
      if (!(str=safe_substitute(fld->rvalue,vars))) return 0;
      bzero(fld->rvalue,strlen(fld->rvalue));
      free(fld->rvalue);
      fld->rvalue=str;
    }
    fld=fld->next;
  }
  return 1;
}

static int
get_passwd(int fd, char *pass, struct termio *orig, struct termio *raw)
{
  char *i, c, *end;
  char *err="Too many characters in password";
  if (raw && ioctl(fd,TCSETAF,raw))
  {
    fprintf(stderr,"\nI/O error setting terminal parameters\n");
    return -1;
  }
  i=pass;
  end=pass+MAX_PASS;
  while (i<end) switch (read(fd,i,1))
  {
  case 1:
    c=*i;
    if (((c & 0x7f)==ASCII_DELETE) || ((c & 0x7f)==ASCII_BACKSPACE))
    {
      if (i>pass) --i;
      break;
    }
    if ((c & 0x7f)>=' ')
    {
      ++i;
      break;
    }
    if ((c!='\n') && (c!='\t') && (c!='\r'))
    {
      err="Illegal character";
      end=i;
      break;
    }
  case 0:
    err=0;
    end=i;
    break;
  default:
    err="I/O error while reading password.";
    end=i;
  }
  *i=c=0;
  if (orig) ioctl(fd,TCSETAF,orig);
  if (!err) return 0;
  fprintf(stderr,"\n%s\n",err);
  return -1;
}

static int
ask_passwd(char *pass, int have_tty)
{
  struct termio  term_orig, term_raw;
  int            cnt=MAX_PASS_ATTEMPTS;
  if (have_tty)
  {
    if (ioctl(STDIN_FILENO,TCGETA,&term_orig))
    {
      fprintf(stderr,"I/O error getting terminal parameters\n");
      return ERR_USER_PASS;
    }
    memcpy(&term_raw, &term_orig, sizeof(term_orig));
    term_raw.c_iflag&=~(INLCR|ICRNL|IUCLC|ISTRIP|IXON|BRKINT);
    term_raw.c_lflag&=~(ICANON|ISIG|ECHO);
    term_raw.c_cc[VTIME]=1;
    term_raw.c_cc[VMIN]=1;
  }
  while (cnt>0)
  {
    if (have_tty)
    {
      fprintf(stderr,"%s","Enter your new mysql password: ");
      fflush(0);
    }
    if (get_passwd(STDIN_FILENO,pass,have_tty ? &term_orig : 0,
                   have_tty ? &term_raw : 0))
      return ERR_USER_PASS;
    if (have_tty)
    {
      char again[MAX_PASS+1];
      int  cmp;
      fprintf(stderr,"\n%s","Retype your new mysql password: ");
      fflush(0);
      if (get_passwd(STDIN_FILENO,again,&term_orig,&term_raw))
        return ERR_USER_PASS;
      fprintf(stderr,"\n");
      cmp=strncmp(pass,again,MAX_PASS);
      bzero(again,MAX_PASS);
      if (!cmp) return STATUS_CONTINUE;
      fprintf(stderr,"%s\n","Passwords don't match.");
      --cnt;
    }
    else return STATUS_CONTINUE;
  }
  fprintf(stderr,"%s\n","Too many unsuccessfull attempts.");
  return ERR_USER_PASS;
}

static int
read_root_passwd(char *pass)
{
  int fd;
  int ret;
  if ((fd=open(ROOT_PASSWD_FILE,O_RDONLY))<0)
  {
    fprintf(stderr,"%s\n","Cannot open file with supervisor password.");
    return ERR_ROOT_PASS;
  }
  ret=get_passwd(fd,pass,0,0);
  close(fd);
  return ret ? ERR_ROOT_PASS : STATUS_CONTINUE;
}

int
update_pass (variable_t **vars, char *target, cfg_field_t *primary,
             cfg_field_t *fields)
{
  MYSQL       db;
  int         ret=STATUS_CONTINUE, have_tty;
  char        user_passwd[MAX_PASS+1];
  char        root_passwd[MAX_PASS+1];
  const char *user;
  have_tty=isatty(STDIN_FILENO);
  ret=read_root_passwd(root_passwd);
  if (ret==STATUS_CONTINUE) ret=db_mysql_open(&db,root_passwd);
  if (ret==STATUS_CONTINUE) ret=ask_passwd(user_passwd,have_tty);
  if (ret==STATUS_CONTINUE) if (target)
  {
    textrope_t *tr_pass=0;
    char       *str_pass=0;
    if (!textrope_esc(&tr_pass,user_passwd,'\\',"\\\'\"")) ret=ERR_SHORT_MEM;
    if ((ret==STATUS_CONTINUE) && !(str_pass=textrope_dup(tr_pass)))
      ret=ERR_SHORT_MEM;
    if ((ret==STATUS_CONTINUE) && (set_var(vars,VAR_PASSWD,str_pass)
        || !substitute_fld(*vars,primary) || !substitute_fld(*vars,fields)))
      ret=ERR_SHORT_MEM;
    textrope_purge(&tr_pass);
    if (str_pass)
    {
      bzero(str_pass,strlen(str_pass));
      free(str_pass);
    }
    if (ret==STATUS_CONTINUE)
      ret=db_mysql_table_passwd(&db,target,primary,fields,have_tty);
    else if (ret==ERR_SHORT_MEM)
      fprintf(stderr,"%s\n","Memory too short to complete the operation");
  }
  else
  {
    user=get_var(*vars,VAR_LOGIN);
    if (!user || !*user)
    {
      fprintf(stderr,"%s\n","internal error");
      ret=ERR_PARAM;
    }
    if (ret==STATUS_CONTINUE)
      ret=db_mysql_db_passwd(&db,user,user_passwd,have_tty);
  }
  db_mysql_close(&db);
  bzero(root_passwd,MAX_PASS);
  bzero(user_passwd,MAX_PASS);
  return ret==STATUS_CONTINUE ? STATUS_DONE : ret;
}
