/*
 * Copyright (C) 2002 Bartosz Lis <bartoszl@ics.p.lodz.pl>
 * This is the main module
 */

#define _GNU_SOURCE

#include "vars.h"
#include "textrope.h"

#include <string.h>
#include <stdlib.h>
#include <ctype.h>

static void
drop(variable_t **var, int purge)
{
  variable_t *vdel;
  vdel=*var;
  *var=(*var)->next;
  if (vdel->name)
  {
    if (purge) bzero(vdel->name,strlen(vdel->name));
    free(vdel->name);
  }
  if (vdel->value)
  {
    if (purge) bzero(vdel->value,strlen(vdel->value));
    free(vdel->value);
  }
  free(vdel);
}

const char *
get_var(variable_t *vars, const char *name)
{
  if (!name || !*name) return 0;
  while (vars) if (strcmp(vars->name,name)) vars=vars->next;
  else return vars->value;
  return 0;
}

const char *
get_var_len(variable_t *vars, const char *name, size_t len)
{
  if (!name || !*name) return 0;
  while (vars) if (strncmp(name,vars->name,len)) vars=vars->next;
  else return vars->value;
  return 0;
}

int
set_var(variable_t **vars, const char *name, const char *value)
{
  variable_t *vset=0;
  char       *val;
  if (!vars || !name || !*name) return 1;
  while (*vars && strcmp((*vars)->name,name)) vars=&((*vars)->next);
  if (*vars) vset=*vars;
  else
  {
    if (!(vset=(variable_t *)malloc(sizeof(variable_t)))) return -1;
    if (!(vset->name=strdup(name)))
    {
      free(vset);
      return -1;
    }
    vset->next=0;
  }
  if (!value) value="";
  if (!(val=strdup(value)))
  {
    if (!*vars)
    {
      free(vset->name);
      free(vset);
    }
    return -1;
  }
  if (*vars) free((*vars)->value); else *vars=vset;
  vset->value=val;
  return 0;
}

int
drop_var(variable_t **vars, const char *name)
{
  if (!vars || !name || !*name) return 1;
  while (*vars) if (strcmp((*vars)->name,name)) vars=&((*vars)->next);
  else
  {
    drop(vars,0);
    return 0;
  }
  return 1;
}

int
purge_var(variable_t **vars, const char *name)
{
  if (!vars || !name || !*name) return 1;
  while (*vars) if (strcmp((*vars)->name,name)) vars=&((*vars)->next);
  else
  {
    drop(vars,1);
    return 0;
  }
  return 1;
}

void
drop_vars(variable_t **vars)
{
  if (vars) while (*vars) drop(vars,0);
}

void
purge_vars(variable_t **vars)
{
  if (vars) while (*vars) drop(vars,1);
}

static char *
textrope_substitute(textrope_t **tr, const char *str, variable_t *vars)
{
  const char *p, *q;
  char        c, delim;
  int         ok=1;
  q=str;
  while (ok && q && *(p=q)) if (!(q=strchr(p,':')))
    ok=textrope_append(tr,0,p);
  else if ((p==q) || (ok=textrope_append(tr,q-p,p)))
  {
    if ((delim=*++q)=='{') ++q; else delim=0;
    c=*(p=q);
    if (isalpha(c) || (c=='_')) do c=*++q; while (isalnum(c) || (c=='_'));
    if (delim && ((c!='}') || (p==q)))
    {
      ok=0;
      break;
    }
    if (p==q)
    {
      if (c==':') ++q;
      ok=textrope_append(tr,1,&c);
    }
    else ok=textrope_append(tr,0,get_var_len(vars,p,q-p));
    if (delim) ++q;
  }
  c=0;
  return ok ? textrope_dup(*tr) : 0;
}

char *
substitute(const char *str, variable_t *vars)
{
  textrope_t *tr=0;
  char       *ret;
  if (!str) return 0;
  ret=textrope_substitute(&tr,str,vars);
  textrope_destroy(&tr);
  return ret;
}

char *
safe_substitute(const char *str, variable_t *vars)
{
  textrope_t *tr=0;
  char       *ret;
  if (!str) return 0;
  ret=textrope_substitute(&tr,str,vars);
  textrope_purge(&tr);
  return ret;
}
